#!/usr/bin/perl

# Program album.pl

# (c) Gary C. Kessler, 2016-2018

# PictureAlbum is a simple program inspired by Picassa and PictureReporter
# to take directories of pictures and create an HTML-formatted page.

# ********************************************************
# MAIN PROGRAM BLOCK
# ********************************************************

use File::Copy;

# Initialize global variables

$version = "4.4";
$build_date = "15 March 2019";
$build_year = substr($build_date,-4);

$verbose = 0;     # Verbose mode is off (0) by default; turn on (1) with -v switch
$albumroot = "";   # Album root directory set with -r switch
$mkthumb = 1;  # Make thumb option set with -t parameter; default to YES
$album_out = "index.html"; # Default output file; override with -o switch

print "\nPicture Album v$version - Gary C. Kessler ($build_date)\n\n";

# Parse command line switches.
# If user has supplied the -h, -v, or an illegal switch, the subroutine
# returns a value of 1 and the program stops.

if (parsecommandline ())
  { exit 1; };

# Read .ini file, if present

if (-e "album_files/album.ini")
    {
    open (INPUT,"<","album_files/album.ini");
    $line = <INPUT>;
    close INPUT;
    chomp ($line);
    ($picsPerRow,$height,$width)=split(/\,/,$line);
    }
  else
    {
    $picsPerRow = 4;
    $height = 200;
    $width = 200;
    }

if ($verbose == 1)
  {
  print "** Images/row = $picsPerRow, image height = $height, image width $width\n";
  print "** Album file: $albumroot/$album_out  Album name: $album_name\n\n";
  }

# Set up thumbnail creation and subdirectory

if ($mkthumb == 1)
  {
  use Image::Magick;

# Clear out thumbnail subdirectory by deleting it (if it exists) and then re-creating it

  $thumbroot = $albumroot . "/thumbnails";
  if (-e $thumbroot)
    { system ("rm -rf $thumbroot"); }
  mkdir $thumbroot;
  copy ("album_files/index.html",$thumbroot);

  $thumb_size = $width . "x" . $height;

  if ($verbose == 1)
    { print "** Thumbnail directory created. Thumb size = $thumb_size\n\n"; }
  }

$num_dirs = 0;
$num_files = 0;

# Copy the generic non-photo icon file into the $albumroot directory

$iconfile = $albumroot . "/non-photo.png";
unless (-e $iconfile)
  {
  copy ("album_files/non-photo.png",$iconfile); 
  if ($verbose == 1)
    { print "** File \"non-photo.png\" written to album root directory\n\n"; }
  }

# Open the album's photo root directory and get the name of all files
# in that directory

opendir (IMD, $albumroot) || die ("Cannot open directory $albumroot");
@dir_list = sort (readdir (IMD));
closedir (IMD);

# Start building report's HTML code

open (REPORT,">",$albumroot."/".$album_out);
print REPORT "<html>\n<head>\n<title>$album_name</title>\n<link rel=\"icon\" href=\"favicon.ico\" type=\"image/x-icon\" />\n</head>\n<body>\n";
print REPORT "<br>\n<h2 align=\"center\">PHOTO ALBUM: $album_name</h2>\n";

print REPORT "<table cellspacing=5 cellpadding=5 align=center>\n<tr>\n<td>\n";
for ($i=0;$i<=4;$i++)
  {
  if ($albumInfo[$i] ne "")
    { print REPORT "$albumInfo[$i]<br>\n"; }
  }
print REPORT "</td>\n</tr>\n";
print REPORT "</table>\n";

print REPORT "<p align=\"center\">\n";
print REPORT "<b>Click on a thumbnail to see the full-size image; go back a page to return to this view.<br>\n";
print REPORT "Use shift-click on a thumbnail to open it in a new tab or window.</b></p>\n";

print REPORT "<a name=\"top\" />\n<h2 align=\"center\">List of Photo Sets</h2>\n";
print REPORT "<table cellspacing=5 cellpadding=5 align=center>\n<tr>\n<td>\n<ul>\n";

# Open a temporary file for output so that we can use main file for table of contents

$reportx = $albumroot."/album.tmp";
open (REPORTX,">",$reportx);

# Now, cycle through all of the files in the album root directory. We
# ignore files and open all directories (except . and ..)
# Also skip the thumbroot directory, if it exists

foreach $dir (@dir_list)
  {
  unless ( ($dir eq ".") || ($dir eq "..") || ($dir eq "thumbnails") )
    {

# We are now in one of the photo root subdirectories. Get the list of files
# in this directory.

    opendir (IMD, $albumroot."/".$dir);
    @file_list = sort (readdir (IMD));
    $list_size = @file_list;
    closedir (IMD);

# This section is a bit if a kludge. An "empty" directory will contain two entries, namely "." and
# "..". In addition, directories created on a Mac may contain ".DS_Store" and Windows systems may have
# "Thumbs.db". We want to skip all of these files!

# $skipsize = 2, which is the number of directory entries that still signify an "empty" directory.
# Check directories to see if they also have .DS_Store and/or Thumbs.db, and increment $skipsize
# as necessary.

# Also, skip over index.html if it is present!

   $skipsize = 2;
   if ($list_size > 2)
     {
     foreach $file (@file_list)
       {
       if ($file eq ".DS_Store") { $skipsize++; }
       if ($file eq "Thumbs.db") { $skipsize++; }
       if ($file eq "index.html") { $skipsize++; }
       }
     }

# Skip empty directories (i.e., process only if # files is greater than $skipsize).

    if ($list_size > $skipsize)
      {

# Copy index.html to the directory so that the contents cannot be listed

    $dir_index = $albumroot."/".$dir."/index.html";
    copy ("album_files/index.html",$dir_index);

# Create a thumbnail subdirectory

    if ($mkthumb == 1)
      {
      $thumb_dir_name = $albumroot."/thumbnails/".$dir;
      $thumb_dir_index = $thumb_dir_name."/index.html";
      mkdir $thumb_dir_name;
      copy ("album_files/index.html",$thumb_dir_index);
      }

# Add directory name to the Contents section of the report and write actual information into temp file

      print REPORT "<li> <a href=\"#$dir\">$dir</a></li>\n";

      print REPORTX "<a name=\"$dir\" />\n";
      print REPORTX "<h3 align=\"center\">$dir</h3><br>\n";
      print REPORTX "<table cellspacing=5 cellpadding=5 align=center>\n";
      print REPORTX "<tr>\n";
      $num_dirs++;
      $i = 0;

# Cycle through all of the files in the photo subdirectory (except . and ..) and prepare the output.
# (NOTE that we only get inside this loop for directories

      foreach $file (@file_list)
        {
        unless ( ($file eq ".") || ($file eq "..") || ($file eq ".DS_Store") || ($file eq "Thumbs.db")  || ($file eq "index.html"))
          {
          $num_files++;
          if ($verbose == 1)
            { print "** $dir: $file\n"; }
          if ($i%$picsPerRow == 0 && $i != 0)
            {
            print REPORTX "</tr>\n<tr>\n";
            }
          $i++;
          print REPORTX "<td>\n";

          thumbnail();

          print REPORTX "Picture #$i</td>\n";
          }
        }

      if ($i%$picsPerRow != 0)
        { print REPORTX "</tr>\n"; }
      print REPORTX "</table>\n";
   print REPORTX "<h5 align=center><a href=\"#top\">Return to top</a></h5>\n<br>\n";
      }
    }
  }

print REPORT "</ul>\n</td>\n</tr>\n";
print REPORT "</table>\n";

# Copy REPORTX temporary contents to REPORT

close $reportx;
open (REPORTX,"<",$reportx);

while (not (eof REPORTX))
  {
  $newline = <REPORTX>;
  print REPORT $newline;
  } ;

close REPORTX;
unlink ($reportx);

print REPORT "<center>\n<h3>=== END OF ALBUM ===</h3>\n";

($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = gmtime (time);
$year += 1900;
$mon += 1;
printf (REPORT "<font size=-1>\n<center>Album Compiled: %02d/%02d/%4d %02d:%02d UTC (%d photo sets with %d photos)</center>\n</font>\n", $mon, $mday, $year,$hour,$min,$num_dirs,$num_files);

print REPORT "<h5>PictureAlbum v.$version &copy; Gary Kessler Associates, $build_year</h5>\n</center>\n";
print REPORT "</body>\n</html>\n";
close (REPORT);

print "\nDone! $num_dirs directories and $num_files files processed!\n\n";


# ********************************************************
# *****           SUBROUTINE BLOCKS                  *****
# ********************************************************

# ********************************************************
# help_text
# ********************************************************

# Display the help file

sub help_text
{
print<< "EOT";
Program usage:
        album.pl [-r album_root] [-o output] [-a album_info] [-v] [-t]
        album.pl [-h]

 where: -r is the directory with the picture files
           *** This must be the first parameter for proper operation!! ***
        -o is the output file [default = index.html]
        -a is the file containing the name of and identifying information
           about the album
        -t indicates that thumbnail subdirectory already exists
        -v turns on verbose output
        -h prints this help file

If the -r or -a switches are missing, the program will ask for the
information.

If you provide a file containing album information, it can have five lines
containing any comments that you would like. You will be prompted for all
five lines; if you create a file, be sure that it has five lines.

EOT
return;
}

# ********************************************************
# parsecommandline
# ********************************************************

# Parse command line for file name, if present. Query
# user for any missing information

# Return $state = 1 to indicate that the program should stop
# immediately (switch -h)

sub parsecommandline
{
my $state = 0;
my $album_info = "";
my $i;

# Parse command line switches ($ARGV array of length $#ARGV)

if ($#ARGV >= 0)
  { 
  for ($i = 0; $i <= $#ARGV; $i++)
    {
    PARMS:
      {
      $ARGV[$i] eq "-r" && do
         {
         $albumroot = $ARGV[$i+1];
         $i++;
         last PARMS;
         };
      $ARGV[$i] eq "-o" && do
         {
         $album_out = $ARGV[$i+1];
         $i++;
         last PARMS;
         };
      $ARGV[$i] eq "-a" && do
         {
         $album_info = $ARGV[$i+1];
         $i++;
         last PARMS;
         };
      $ARGV[$i] eq "-t" && do
         {
         $mkthumb = 0;
         last PARMS;
         };
      $ARGV[$i] eq "-v" && do
         {
         $verbose = 1;
         last PARMS;
         };
      $ARGV[$i] eq "-h" && do
         {
         help_text();
         $state = 1;
         return $state;
         };

      do
         {
         print "Invalid parameter \"$ARGV[$i]\".\n\n";
         print "Usage: report.pl [-r album_root] [-o output] [-a album_info] [-v] [-t]\n";
         print "       report.pl [-h]\n\n";
         $state = 1;
         return $state;
         };
      };
    };
  };

# Read files or prompt for missing information

# First get album root directory

if ($albumroot eq "")
  {
  print "\nEnter the album root directory: \n";
  chomp ($albumroot = <STDIN>);
  }

# Now check album name and information

if ($album_info ne "")
    {
    open (INPUT, "<", $albumroot."/".$album_info) || die ("Cannot open album file $albumroot/$album_info");
    chomp ($album_name = <INPUT>);
    chomp ($albumInfo[0] = <INPUT>);
    chomp ($albumInfo[1] = <INPUT>);
    chomp ($albumInfo[2] = <INPUT>);
    chomp ($albumInfo[3] = <INPUT>);
    chomp ($albumInfo[4] = <INPUT>);
    close (INPUT);
    }
  else
    {
    print "\nEnter album name: ";
    chomp ($album_name = <STDIN>);

    print "\nEnter album information (five lines or leave blank):\n";
    for ($i=1;$i<=5;$i++)
      {
      print "\n  Enter line $i: ";
      chomp ($albumInfo[$i-1] = <STDIN>);
      }

    print "\n\nWriting album information to '$albumroot/album_info.txt' file...\n\n";
    open (OUTPUT, ">", $albumroot."/album_info.txt");
    print OUTPUT "$album_name\n";
    for ($i=0;$i<=4;$i++)
      { print OUTPUT "$albumInfo[$i]\n"; }
    close (OUTPUT);
    }

return $state;
}

# ********************************************************
# thumbnail
# ********************************************************

# Create a thumbnail image if an image file
# Display the thumbnail of an image file or else a generic icon

sub thumbnail
{
my ($file_ext) = $file =~ /(\.[^.]+)$/;
my $s_file, $t_file, $thumb_file, $thumb_size, $x;

$file_ext = lc($file_ext);

# If the file extension indicates an obvious image file, then create a thumbnail image, place into the thumbnail subdirectory, and
#  create HTML code to display a thumbnail of the image;
# otherwise, display a generic file icon

if ($file_ext eq ".jpg" || $file_ext eq ".jpeg" || $file_ext eq ".png" || $file_ext eq ".gif" || $file_ext eq ".bmp")
    {
    $x = length ($file) - length ($file_ext);
    $thumb_file = substr ($file,0,$x) . "_t" . $file_ext;

    $s_file = $albumroot . "/" . $dir . "/" . $file;
    $t_file = $albumroot . "/thumbnails/" . $dir . "/" . $thumb_file;

    my $image = Image::Magick->new;
    $image->Read($s_file);
    $image->Thumbnail($thumb_size);
    $image->Write($t_file);

    print REPORTX "<a href=\"$dir/$file\"><img src=\"thumbnails/$dir/$thumb_file\" height=$height width=$width></a><br>\n";
    }
  else
    {
    print REPORTX "<a href=\"$dir/$file\"><img src=\"non-photo.png\" height=$height width=$width></a><br>\n";
    } 

return;
}
